/*
 * Copyright (c) 2016 TomTom International B.V.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/* Version history:
 *
 * 1.0 - initial release
 * 1.1 - fix socketSend() function to send entire message if buffers are full
 *       don't select() for write-ready in main loop to save CPU
 *       some minor fixes and cleanup
 * 1.2 - improve command line parsing
 *       allow using a client socket for the applink connection, selectable via command line parameter
 * 1.3 - make the code more C-friendly, so it compiles with gcc as well (use -std=gnu99)
 * 1.4 - allow using a device (i.e. /dev/...) for the applink connection
 * 1.5 - explicitly set device given by -d parameter to raw mode
 * 1.6 - stability improvements, resource leak fixes, minor code refactoring
 * 1.7 - implement commands #0 (keep-alive) and #6 (information), set protocol version to 1.0
 *
 * IMPORTANT REMARK:
 * This code is single-threaded and uses blocking writes in socketSend(), which means it is possible
 * that the entire program locks up if one client malfunctions by remaining connected without reading
 * incoming data from its socket. Therefore, it is not production-ready.
 * To properly handle this and remain single-threaded, non-blocking writes should be used and select()
 * should test if each socket is writable before writing. This adds complexity and does not help with
 * understanding the applink protocol, which is why it is not implemented here.
 */


//lint -save -e451
//lint -save -e471
//lint -save -e514
//lint -save -e573
//lint -save -e665
//lint -save -e676
//lint -save -e701
//lint -save -e702
//lint -save -e703
//lint -save -e713
//lint -save -e717
//lint -save -e732
//lint -save -e734
//lint -save -e737
//lint -save -e740
//lint -save -e749
//lint -save -e750
//lint -save -e1784

//lint -esym(529,target)
//lint -esym(530,socket_id)
//lint -esym(641,__socket_type)
//lint -esym(715,aSignal)
//lint -esym(771,fd)
//lint -esym(818,aBuf)


// TODO: Adapt applink proxy code for WinSim

#include "hall_std_if.h"
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <errno.h>
#include <resolv.h>
#include <signal.h>
#include <termios.h>
#include <iomanip>
#include <string>

#include "applinkproxy.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_APPHMI_NAVIGATION_HALL
#define ETG_I_TRACE_CHANNEL       TR_TTFIS_APPHMI_NAVIGATION
#define ETG_I_TTFIS_CMD_PREFIX    "APPHMI_NAVIGATION_"
#define ETG_I_FILE_PREFIX         App::Core::ApplinkProxy::applinkproxy::
#include "trcGenProj/Header/applinkproxy.cpp.trc.h"
#endif

#define LOG_ERROR 0
#define LOG_WARN 1
#define LOG_INFO 2
#define LOG_DEBUG 3
#define LOG_VERBOSE 4


// Define LOGLEVEL as LOG_DEBUG to also log all message headers, or as LOG_VERBOSE to log all bytes.
#ifndef LOGLEVEL
#ifdef DEBUG
// Compatibility in case DEBUG is defined
#define LOGLEVEL LOG_VERBOSE
#else
#define LOGLEVEL LOG_INFO
#endif
#endif


// AppLink protocol version
#define PROTOCOL_VERSION_MAJOR 1
#define PROTOCOL_VERSION_MINOR 0

#define MAX_CONNECTIONS 100
#define MAX_DEVICE_ID_LEN 64

#define bool int
#define false 0
#define true 1


static int pfd[2];
void output(const char* fmt, va_list ap)
{
#ifdef APPLINKPROXY_STANDALONE
   vprintf(fmt, ap);
   fflush(stdout);
#endif
}


#define LOGE(...) LOG(LOG_ERROR,   __VA_ARGS__)
#define LOGW(...) LOG(LOG_WARN,    __VA_ARGS__)
#define LOGI(...) LOG(LOG_INFO,    __VA_ARGS__)
#define LOGD(...) LOG(LOG_DEBUG,   __VA_ARGS__)
#define LOGV(...) LOG(LOG_VERBOSE, __VA_ARGS__)

#define LOG(level, fmt, ...) if (level <= LOGLEVEL) {logmsg("%c %s@%d: "fmt, "EWIDV"[level], __FUNCTION__, __LINE__, ##__VA_ARGS__);}

void logmsg(const char* aFmt, ...)
{
   va_list ap;
   va_start(ap, aFmt);
   output(aFmt, ap);
   va_end(ap);
}


// Handler to catch SIGINT (so proper cleanup can be done).
volatile sig_atomic_t gInterrupted = false;
void sigHandler(int aSignal)
{
   gInterrupted = true;
}


// Puts the given file descriptor in raw mode.
// Returns false on failure.
bool makeDeviceRaw(int aFd)
{
   struct termios termios_p;
   if (tcgetattr(aFd, &termios_p) < 0)
   {
      ETG_TRACE_FATAL(("applinkproxy::tcgetattr() failed: %s ", strerror(errno)));
      return false;
   }
   termios_p.c_cc[VMIN]  =  1;
   termios_p.c_cc[VTIME] =  0;
   termios_p.c_lflag &= ~(ECHO | ICANON | ISIG | ECHOE | ECHOK | ECHONL);
   termios_p.c_oflag &= ~(OPOST);
   if (tcflush(aFd, TCIOFLUSH) < 0)
   {
      ETG_TRACE_FATAL(("applinkproxy::tcioflush() failed: %s ", strerror(errno)));
      return false;
   }
   cfmakeraw(&termios_p);
   if (tcsetattr(aFd, TCSANOW, &termios_p) < 0)
   {
      ETG_TRACE_FATAL(("applinkproxy::tcsetattr() failed: %s ", strerror(errno)));
      return false;
   }
   return true;
}


// Sets socket to blocking or non-blocking mode.
// Returns false on failure.
bool setBlockingModeEnabled(int aFd, bool aIsBlocking)
{
   ETG_TRACE_SYS(("applinkproxy::setBlockingModeEnabled(aIsBlocking: %d)", aIsBlocking));

   int flags = fcntl(aFd, F_GETFL, 0);
   if (flags < 0)
   {
      ETG_TRACE_FATAL(("applinkproxy::Failed to get socket flags: %s ", strerror(errno)));
      return false;
   }

   //Check to verify if the descriptor is already in requested(blocking or nonblocking) mode
   if (((flags & O_NONBLOCK) == 0) == aIsBlocking)
   {
      ETG_TRACE_FATAL(("applinkproxy::Socket was already in %sblocking mode ", (aIsBlocking ? "" : "non-")));
      return true;
   }
   int ret = fcntl(aFd, F_SETFL, flags ^ O_NONBLOCK);
   return ret >= 0;
}


// Create new server socket. Optionally set a different backlog (# of pending connections).
// Returns new socket descriptor, or -1 on failure.
int createServerSocket(int aPort, int aBacklog)
{
   // Create socket
   int fd = socket(AF_INET, SOCK_STREAM, 0);
   if (fd < 0)
   {
      ETG_TRACE_FATAL(("applinkproxy::Failed to open server socket: %s ", strerror(errno)));
      return -1;
   }
   // Enable binding while socket is still in time_wait state (from previous execution)
   int option = 1;
   if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char*) &option, sizeof(option)) < 0)
   {
      ETG_TRACE_SYS_MIN(("applinkproxy::Failed to set socket option %s ", strerror(errno)));
      close(fd);
      return -1;
   }
   // Bind socket
   struct sockaddr_in server_addr;
   memset(&server_addr, 0, sizeof(server_addr));
   server_addr.sin_family = AF_INET;
   server_addr.sin_addr.s_addr = INADDR_ANY;
   server_addr.sin_port = htons(aPort);
   if (bind(fd, (struct sockaddr*) &server_addr, sizeof(server_addr)) < 0)
   {
      ETG_TRACE_SYS_MIN(("applinkproxy::Failed to bind server socket at port %d: %s ", aPort, strerror(errno)));
      close(fd);
      return -1;
   }
   // Start listening for incoming connections
   if (listen(fd, aBacklog) < 0)
   {
      close(fd);
      fd = -1;
      ETG_TRACE_SYS_MIN(("applinkproxy::Failed to listen server socket %s ", strerror(errno)));
   }
   return fd;
}


// Accepts an incoming client connection on the given server socket.
// Returns new socket descriptor, or -1 on failure.
int acceptClientSocket(int aServerSocket)
{
   struct sockaddr_in client_addr;
   socklen_t client_addr_len = sizeof(client_addr);
   int fd = accept(aServerSocket, (struct sockaddr*) &client_addr, &client_addr_len);
   if (fd < 0)
   {
      ETG_TRACE_FATAL(("applinkproxy::Failed to accept client connection: %s ", strerror(errno)));
      return -1;
   }
   return fd;
}


// Create new client socket, connecting to aHost:aPort.
// Returns new socket descriptor, or -1 on failure.
int createClientSocket(const char* aHost, int aPort)
{
   // Lookup host for IPv4 TCP socket
   char port[6]; // 5 digits + NULL
   snprintf(port, sizeof(port), "%d", aPort);
   struct addrinfo hints;
   memset(&hints, 0, sizeof(hints));
   hints.ai_family = AF_INET; // IPv4
   hints.ai_socktype = SOCK_STREAM; // TCP socket
   hints.ai_flags = 0;
   hints.ai_protocol = 0;
   struct addrinfo* result = NULL;
   int errCode = getaddrinfo(aHost, port, &hints, &result);
   if (errCode != 0)
   {
      ETG_TRACE_FATAL(("applinkproxy::Failed to lookup host: %s ", gai_strerror(errCode)));
      return -1;
   }
   // Connect to the first successful IP address
   int fd;
   struct addrinfo* resultPointer = NULL;
   for (resultPointer = result; resultPointer != NULL; resultPointer = resultPointer->ai_next)
   {
      fd = socket(resultPointer->ai_family, resultPointer->ai_socktype, resultPointer->ai_protocol);
      if (fd == -1)
      {
         continue;
      }
      if (connect(fd, resultPointer->ai_addr, resultPointer->ai_addrlen) != -1)
      {
         break; // Success
      }
      close(fd);
      fd = -1;
   }
   // Cleanup
   freeaddrinfo(result);
   if (resultPointer == NULL)   // No IP address succeeded
   {
      ETG_TRACE_ERR(("applinkproxy::Failed to connect to %d:%s! ",  aPort , aHost));
      ETG_TRACE_ERR(("applinkproxy::Failed to connect err: %d:%s ",  errno , strerror(errno)));
   }
   return fd;
}


// Write 'aLen' bytes from circular buffer 'aBuf' of total size 'aBufSize' starting at 'aStart' to socket 'aFd'
// Can be used for regular buffers too, just specify aStart=0 and aLen=aBufSize.
// Blocks when socket's write buffer is full.
// Returns false on failure.
bool socketSend(int aFd, const char* aBuf, int aBufSize, int aStart, int aLen)
{
   ETG_TRACE_COMP(("applinkproxy::socketSend(bufSize: %d start: %d len: %d)", aBufSize, aStart, aLen));

   do
   {
      aStart %= aBufSize;
      int size = aLen;
      if (aLen > aBufSize - aStart)
      {
         size = aBufSize - aStart;
      }
      int written = write(aFd, &aBuf[aStart], size);
      if (written < 0)
      {
         if (errno != EAGAIN && errno != EWOULDBLOCK)
         {
            ETG_TRACE_FATAL(("applinkproxy::Error writing to client socket: %s ", strerror(errno)));
            return false;
         }
         // Nothing was written because the write buffer is full
         written = 0;
         // Wait until socket is ready for writing again
         fd_set write_fd_set;
         struct timeval selectTimeout = {10, 0}; // 10 seconds
         FD_ZERO(&write_fd_set);
         FD_SET(aFd, &write_fd_set);
         int sel = select(aFd + 1, (fd_set*) NULL, &write_fd_set, (fd_set*) NULL, &selectTimeout);
         if (sel < 0)
         {
            ETG_TRACE_FATAL(("applinkproxy::Error in select: %s ", strerror(errno)));
            return false;
         }
      }
      aStart += written;
      aLen -= written;
   }
   while (aLen > 0);
   return true;
}


// Send an applink message to socket 'aFd'.
// Returns false on failure.
bool applinkSend(int aFd, int aCommand, int aTarget, int aConnId, const char* aData, int aDataLen)
{
   ETG_TRACE_COMP(("applinkproxy::Sending message to SOCKS service: dataLen=%d command=%d target=%d connId=%d ",
                   aDataLen, aCommand, aTarget, aConnId));

   char outData[aDataLen + 7] =
   {
      (uint8_t)(aDataLen >> 8), (uint8_t)(aDataLen & 0xff),
      (uint8_t)((aCommand & 0x1f) | (aTarget << 5)),
      (uint8_t)((aConnId >> 24) & 0xff), (uint8_t)((aConnId >> 16) & 0xff), (uint8_t)((aConnId >> 8) & 0xff), (uint8_t)(aConnId & 0xff)
   };

   if (aDataLen > 0)
   {
      memcpy(outData + 7, aData, aDataLen);
   }

   if (etg_bIsTraceActiveShort(((0xFFFFu & (etg_tU16)(TR_CLASS_APPHMI_NAVIGATION_HALL)) << 16) | (etg_tU16)(ETG_LEVEL_USER_4)) == (etg_tBool)TRUE)
   {
      char line[8];
      for (int i = 0, l = 0; l < sizeof(outData); l += 8)
      {
         for (unsigned int j = 0; j < 8; j++)
         {
            line[j] = i < sizeof(outData) ? outData[i++] : 0;
         }
         ETG_TRACE_USR4(("applinkproxy::applinkSend DATA 0x%04x: %02x %02x %02x %02x %02x %02x %02x %02x",
                         l, line[0], line[1], line[2], line[3], line[4], line[5], line[6], line[7]));
      }
   }

   if (!socketSend(aFd, outData, sizeof(outData), 0, sizeof(outData)))
   {
      return false;
   }

   return true;
}


// SOCKS service
#define TARGET_PHONE 0
// SOCKS proxy
#define TARGET_HU 1

enum ConnectionMode {CM_UNKNOWN, CM_SERVER_SOCKET, CM_CLIENT_SOCKET, CM_DEVICE};

#ifdef APPLINKPROXY_STANDALONE

int main(int argc, char* argv[])
{
   logmsg("%s, built %s %s, loglevel %c\n\n", argv[0], __DATE__, __TIME__, "EWIDV"[LOGLEVEL]);

   char* applinkHost;
   char* applinkDeviceFile;
   const char* deviceId = "undefined";
   int applinkPort;
   int socksPort = -1;
   bool cmdLineValid = true;
   enum ConnectionMode connectionMode = CM_UNKNOWN;
   int c;
   while ((c = getopt(argc, argv, "s:c:d:p:i:")) != -1)
   {
      switch (c)
      {
         case 's': // Applink server port
            cmdLineValid &= (connectionMode == CM_UNKNOWN);
            connectionMode = CM_SERVER_SOCKET;
            cmdLineValid &= (sscanf(optarg, "%i", &applinkPort) == 1);
            break;
         case 'c': // Applink client host:port
            cmdLineValid &= (connectionMode == CM_UNKNOWN);
            connectionMode = CM_CLIENT_SOCKET;
            // Split host:port
            cmdLineValid &= ((applinkHost = strtok(optarg, ":")) != NULL);
            if (cmdLineValid)
            {
               char* ptr = strtok(NULL, ":");
               cmdLineValid &= (ptr != NULL) && (sscanf(ptr, "%i", &applinkPort) == 1);
            }
            break;
         case 'd': // Applink device filename
            cmdLineValid &= (connectionMode == CM_UNKNOWN);
            connectionMode = CM_DEVICE;
            applinkDeviceFile = optarg;
            break;
         case 'p': // SOCKS server port
            cmdLineValid &= (sscanf(optarg, "%i", &socksPort) == 1);
            break;
         case 'i': // Device ID
            deviceId = optarg;
            break;
         default:
            LOGE("Unknown option '%c'!\n", (char) c);
            cmdLineValid = false;
      }
   }
   cmdLineValid &= (connectionMode != CM_UNKNOWN);
   cmdLineValid &= (socksPort >= 0);
   if (!cmdLineValid)
   {
      printf("Usage: %s <-s APPLINK_SERVER_PORT|-c APPLINK_CLIENT_HOST:PORT|-d DEVICE_NAME> -p SOCKS_SERVER_PORT [-i DEVICE_ID]\n", argv[0]);
      printf("\n");
      printf("This program runs a SOCKS proxy locally on SOCKS_SERVER_PORT.\n");
      printf("\n");
      printf("It connects to the remote Applink instance using one of these methods:\n");
      printf("  -s  listen for a single Applink client to connect on APPLINK_SERVER_PORT\n");
      printf("  -c  connect as Applink client to APPLINK_CLIENT_HOST:PORT\n");
      printf("  -d  use a device filename for communication (i.e. /dev/...)\n");
      printf("\n");
      printf("The DEVICE_ID is sent to the remote side when Applink is connected, and defaults to 'undefined'.\n");
      printf("\n");
      printf("Example on how to set this up using a reverse forwarded ADB socket:\n");
      printf("1) configure companion app for Bluetooth\n");
      printf("2) adb shell 'echo localhost:9999 > /mnt/sdcard/com.tomtom.mydrive.tcpsocket.txt'\n");
      printf("3) adb reverse tcp:9999 tcp:9999\n");
      printf("4) start this program: %s -s 9999 -p 1080\n", argv[0]);
      printf("5) adb shell am force-stop com.tomtom.mydrive.aivi_dev\n");
      printf("6) start companion app\n");
      printf("7) test if IDX connection works (using netcat): nc -x localhost:1080 localhost 30002\n");
      printf("8) configure your favorite application to connect via the SOCKS proxy @ localhost:1080 and test if it works\n");
      return 1;
   }
#else
int execute_applinkproxy(
   const char* applinkHost,
   int applinkPort,
   const char* applinkDeviceFile,
   const char* localDeviceID)
{
   int socksPort = 1080;
   enum ConnectionMode connectionMode = CM_UNKNOWN;

   if (applinkHost != NULL && strlen(applinkHost) > 0 && applinkPort != 0)
   {
      connectionMode = CM_CLIENT_SOCKET;
   }
   else if (applinkDeviceFile != NULL && strlen(applinkDeviceFile) > 0)
   {
      connectionMode = CM_DEVICE;
   }
#endif

   // Setup Applink connection with SOCKS service.
   int applinkHandle = -1;

   switch (connectionMode)
   {
      case CM_CLIENT_SOCKET:
         // We act as a TCP client for Applink.
         ETG_TRACE_SYS(("applinkproxy::Connecting to Applink server at %d :%s...", applinkPort , applinkHost));
         applinkHandle = createClientSocket(applinkHost, applinkPort);
         break;
      case CM_DEVICE:
         // Use a device (i.e. /dev/...) to communicate.
         ETG_TRACE_SYS(("applinkproxy::Connecting to Applink remote via device %s...", applinkDeviceFile));
         applinkHandle = open(applinkDeviceFile, O_RDWR | O_NOCTTY | O_SYNC);
         if (applinkHandle < 0)
         {
            ETG_TRACE_ERR(("applinkproxy::Failed to open Applink device: %s ", strerror(errno)));
         }
         else if (!makeDeviceRaw(applinkHandle))
         {
            close(applinkHandle);
            ETG_TRACE_FATAL(("applinkproxy::Creating raw applink handle failed"));
            return 1;
         }
         break;
      default:
         ETG_TRACE_ERR(("applinkproxy::default connection mode! %d", connectionMode));
         return 1;
   }
   if (applinkHandle < 0)
   {
      ETG_TRACE_FATAL(("applinkproxy::Invalid applink handle"));
      return 1;
   }

   if (!setBlockingModeEnabled(applinkHandle, false))
   {
      close(applinkHandle);
      applinkHandle = -1;
      ETG_TRACE_FATAL(("applinkproxy::Set block mode on applinkhandle failed"));
      return 1;
   }

   ETG_TRACE_SYS(("applinkproxy::Applink connection established."));

   int ret = run_applink(applinkHandle, socksPort, localDeviceID);

   close(applinkHandle);
   return ret;
}


// Called when the remote device ID is received. aRemoteDeviceId points to a zero-terminated string.
// The pointer aRemoteDeviceId becomes invalid after this function returns, so its data should be copied.
void onRemoteDeviceIdReceived(unsigned char* aRemoteDeviceId)
{
   // Use the remote device ID for i.e. TLS handshake
}


int run_applink(int aApplinkHandle, int aSocksPort, const char* aDeviceId)
{
   ETG_TRACE_SYS(("applinkproxy::Local protocol version: %d.%d", PROTOCOL_VERSION_MAJOR, PROTOCOL_VERSION_MINOR));
   ETG_TRACE_SYS(("applinkproxy::Local device ID: %s", aDeviceId));

   ::std::string serverDeviceId;

   // create server socket for SOCKS clients, typically listening on localhost
   bool isSocksServerCreated = false;
   int socksServerSocket = -1;

   // association between SOCKS client socket handles and connection IDs
   struct
   {
      int socket;
      int id;
   } socket_id[MAX_CONNECTIONS];
   int socket_id_used = 0;

   // next unused connection ID for target 0 (SOCKS service/companion app)
   int nextConnId = 0;

   // circular read buffer for messages from SOCKS service (companion app)
   char aq[65542]; // max size of an Applink message is 65542
   int aqTail = 0; // index where next char is read from
   int aqUsed = 0; // number of used chars in buffer

   // macros to read 8/16/32 bit big-endian values, with wraparound
#define AQREAD1(index) aq[(index) % sizeof(aq)]
#define AQREAD2(index) ((AQREAD1(index) << 8) | AQREAD1(index + 1))
#define AQREAD4(index) ((AQREAD2(index) << 16) | AQREAD2(index + 2))

   // send protocol version and device ID to SOCKS service (information, 4.2.7):
   // {command=6, ConnectionId=-1, data=<0x00 major minor>}
   char versionBuf[] = {0, PROTOCOL_VERSION_MAJOR, PROTOCOL_VERSION_MINOR};

   //TODO uncomment next line to support EIPF18
   applinkSend(aApplinkHandle, 6, TARGET_PHONE, -1, versionBuf, sizeof(versionBuf));

   // {command=6, ConnectionId=-1, data=<0x01 length id_string>}
   int deviceIdLen = strlen(aDeviceId);
   if (deviceIdLen > MAX_DEVICE_ID_LEN)
   {
      deviceIdLen = MAX_DEVICE_ID_LEN;
   }
   char deviceIdBuf[2 + MAX_DEVICE_ID_LEN] = {1, (uint8_t)deviceIdLen};
   memcpy(&deviceIdBuf[2], aDeviceId, deviceIdLen);

   //TODO uncomment next line to support EIPF18
   applinkSend(aApplinkHandle, 6, TARGET_PHONE, -1, deviceIdBuf, 2 + deviceIdLen);

   signal(SIGINT, sigHandler); // Sets gInterrupted to true on SIGINT
   signal(SIGPIPE, SIG_IGN); // Ignore SIGPIPE (when remote end disappears), rely on EPIPE and zero-length read instead

   bool isStopCalled = false;
   while (!gInterrupted)
   {
      ETG_TRACE_COMP(("applinkproxy::tick"));

      // start the SOCKS server socket for clients only after receiving remote device id.
      ETG_TRACE_COMP(("applinkproxy::isSocksServerCreated: %d serverDeviceId: %s", isSocksServerCreated, serverDeviceId.c_str()));
      if (!serverDeviceId.empty() && !isSocksServerCreated)
      {
         socksServerSocket = createServerSocket(aSocksPort, 5);
         if (socksServerSocket == -1)
         {
            ETG_TRACE_FATAL(("applinkproxy::Error in creating server socket: %d - %s", errno, strerror(errno)));
            return 1;
         }

         ETG_TRACE_SYS(("applinkproxy::SOCKS server socket created."));
         isSocksServerCreated = true;

         if (!setBlockingModeEnabled(socksServerSocket, false))
         {
            close(socksServerSocket);
            socksServerSocket = -1;
            ETG_TRACE_FATAL(("applinkproxy::Enabling block mode on server socket failed"));
            return 1;
         }
      }

      // wait for available data on any socket (SOCKS service, server socket or any SOCKS client)
      fd_set read_fd_set;
      struct timeval selectTimeout = {2 * 60, 0}; // 2 minutes
      FD_ZERO(&read_fd_set);
      FD_SET(aApplinkHandle, &read_fd_set);
      if (socksServerSocket != -1)
      {
         ETG_TRACE_USR4(("applinkproxy::FD_SETting socksServerSocket: %d", socksServerSocket));
         FD_SET(socksServerSocket, &read_fd_set);
      }
      FD_SET(pfd[0], &read_fd_set);

      for (int i = 0; i < socket_id_used; i++)
      {
         ETG_TRACE_COMP(("applinkproxy::FD_SETting client socket #%d", i));
         FD_SET(socket_id[i].socket, &read_fd_set);
      }

      int sel = select(FD_SETSIZE, &read_fd_set, (fd_set*) NULL, (fd_set*) NULL, &selectTimeout);
      if (sel < 0)
      {
         if (!gInterrupted)
         {
            ETG_TRACE_ERR(("applinkproxy::Error in select(): %d - %s", errno, strerror(errno)));
         }
         break;
      }
      else if (sel == 0)
      {
         // Timeout expired; send a keep-alive command (4.2.1) to detect half-open connections.
         // The remote side is expected to ignore this message.
         ETG_TRACE_COMP(("applinkproxy::Sending keep-alive..."));

         //TODO uncomment next line to support EIPF18
         applinkSend(aApplinkHandle, 0, TARGET_PHONE, -1, NULL, 0);
      }
      for (int fd = 0; fd < FD_SETSIZE; fd++)
      {
         if (FD_ISSET(fd, &read_fd_set))
         {
            ETG_TRACE_COMP(("applinkproxy::FD_ISSET: %d", fd));

            if (fd == pfd[0])
            {
               ETG_TRACE_COMP(("applinkproxy::Stop called FD_ISSET: %d", fd));
               isStopCalled = true;
            }
            // if a new SOCKS client wants to connect to the server socket:
            else if (fd == socksServerSocket)
            {
               // accept client socket
               int clientSocket = acceptClientSocket(socksServerSocket);
               if (clientSocket < 0)
               {
                  ETG_TRACE_FATAL(("applinkproxy::Connection to client socket failed"));
                  return 1;
               }

               // with "accept" BSD sockets inherit file status flags such as O_NONBLOCK,
               // so explicitly enable blocking mode
               setBlockingModeEnabled(clientSocket, true);
               ETG_TRACE_SYS(("applinkproxy::New SOCKS client: %d", clientSocket));

               // receive SOCKS authentication offer from client
               char buffer[256];
               int bufRead = read(clientSocket, buffer, sizeof(buffer));
               if (bufRead < 0)
               {
                  ETG_TRACE_ERR(("applinkproxy::Error reading from SOCKS client: %s", strerror(errno)));
                  close(clientSocket);
                  continue;
               }

               // Offer must start with '5' and be at least 2 bytes, see RFC1928 Ch3 for details.
               if ((bufRead < 2) || buffer[0] != 5)
               {
                  ETG_TRACE_ERR(("applinkproxy::SOCKS client did not send valid auth request!"));
                  close(clientSocket);
                  continue;
               }

               // check that client offered auth method 0 (no auth)
               bool offeredMethod0 = false;
               unsigned int authMethodsNum = buffer[1];
               if (authMethodsNum > sizeof(buffer) - 2)
               {
                  authMethodsNum = sizeof(buffer) - 2;
               }
               for (unsigned int i = 0; i < authMethodsNum; i++)
               {
                  ETG_TRACE_SYS(("applinkproxy::Client offers auth method %u", i));
                  offeredMethod0 |= (buffer[2 + i] == 0);
               }
               if (!offeredMethod0)
               {
                  ETG_TRACE_ERR(("applinkproxy::SOCKS client did not offer auth method 0 (no auth)!"));
                  close(clientSocket);
                  continue;
               }

               // reply to client with SOCKS authentication acceptance
               char data[] = {5, 0}; // accept auth method 0
               socketSend(clientSocket, data, sizeof(data), 0, sizeof(data));
               ETG_TRACE_SYS(("applinkproxy::SOCKS client authenticated: %d", clientSocket));

               // receive SOCKS client's remote id request from E2E or SOCKS connect command from Traffic/Online search
               bufRead = read(clientSocket, buffer, sizeof(buffer));
               if (bufRead < 0)
               {
                  ETG_TRACE_ERR(("applinkproxy::Error reading from SOCKS client: %s", strerror(errno)));
                  close(clientSocket);
                  continue;
               }

               // Check SOCKS device id authentication
               if (0 == strncmp(buffer, "DeviceId", bufRead))
               {
                  ETG_TRACE_SYS(("applinkproxy::SOCKS client requested for Device Id."));

                  // send local device id
                  ::std::string deviceIdMsg;

                  // Note: Don't change below message it has to be same as in E2ETransportLayerSockV5
                  deviceIdMsg.assign("LId:, RId:");
                  deviceIdMsg.insert(4, aDeviceId);
                  deviceIdMsg.append(serverDeviceId);
                  ETG_TRACE_SYS(("applinkproxy::Local and Remote Device id message: %s", deviceIdMsg.c_str()));
                  socketSend(clientSocket, deviceIdMsg.c_str(), deviceIdMsg.size(), 0, deviceIdMsg.size());

                  // receive SOCKS connect command from client who first requested DeviceId
                  bufRead = read(clientSocket, buffer, sizeof(buffer));
                  if (bufRead < 0)
                  {
                     ETG_TRACE_ERR(("applinkproxy::Error reading from SOCKS client: %s", strerror(errno)));
                     close(clientSocket);
                     continue;
                  }
               }

               // generate new connection ID
               int connId = nextConnId;
               nextConnId++;

               // send message to SOCKS service (connect, 4.2.2):
               // {command=1, ConnectionId=<generated new ID>, data=<command from client>}
               applinkSend(aApplinkHandle, 1, TARGET_PHONE, connId, buffer, bufRead);

               // store association between connection ID and client handle/socket
               if (socket_id_used >= MAX_CONNECTIONS)
               {
                  ETG_TRACE_SYS_MIN(("applinkproxy::Max # of connections exceeded, rejecting client!"));
                  close(clientSocket);
                  continue;
               }
               socket_id[socket_id_used].id = connId;
               socket_id[socket_id_used].socket = clientSocket;
               socket_id_used++;

               // Make client non-blocking
               if (!setBlockingModeEnabled(clientSocket, false))
               {
                  ETG_TRACE_FATAL(("applinkproxy::Setting block mode on client socket failed"));
                  return 1;
               }
               ETG_TRACE_SYS(("applinkproxy::SOCKS client: %d got connId: %d and requested a connection", clientSocket, connId));
            }
            // if some data is received from the SOCKS service:
            else if (fd == aApplinkHandle)
            {
               // append data to read buffer (if the circular buffer wraps around, read the second part next time)
               unsigned int aqHead = aqTail + aqUsed;
               int maxReadSize = sizeof(aq) - aqHead;
               if (aqHead >= sizeof(aq))
               {
                  aqHead -= sizeof(aq);
                  maxReadSize = sizeof(aq) - aqUsed;
               }
               int n = read(aApplinkHandle, &aq[aqHead], maxReadSize);
               if (n < 0)
               {
                  ETG_TRACE_FATAL(("applinkproxy::Error reading from applink socket: %s", strerror(errno)));
                  return 1;
               }
               if (n == 0)
               {
                  ETG_TRACE_ERR(("applinkproxy::Applink connection lost!"));
                  gInterrupted = true;
                  continue;
               }
               ETG_TRACE_COMP(("applinkproxy::Received %d bytes from SOCKS service (aqUsed=%d):", n, aqUsed));
#if (LOGLEVEL >= LOG_VERBOSE)
               for (int i = 0; i < n; i++)
               {
                  if (i != 0 && i % 16 == 0)
                  {
                     logmsg("\n");
                  }
                  logmsg(" %02X", AQREAD1(aqHead + i));
               }
               logmsg("\n ");
               for (int i = 0; i < n; i++)
               {
                  unsigned char c = AQREAD1(aqHead + i);
                  if (i != 0 && i % 70 == 0)
                  {
                     logmsg("\n ");
                  }
                  logmsg("%c", c >= 0x20 && c < 0x7E ? c : '.');
               }
               logmsg("\n");
#endif

               if (etg_bIsTraceActiveShort(((0xFFFFu & (etg_tU16)(TR_CLASS_APPHMI_NAVIGATION_HALL)) << 16) | (etg_tU16)(ETG_LEVEL_USER_4)) == (etg_tBool)TRUE)
               {
                  if (n > 0)
                  {
                     char line[8];
                     for (unsigned int i = 0, l = 0; l < n; l += 8)
                     {
                        for (unsigned int j = 0; j < 8; j++)
                        {
                           line[j] = i < n ? aq[aqHead + i++] : 0;
                        }
                        ETG_TRACE_USR4(("applinkproxy::Received DATA 0x%04x: %02x %02x %02x %02x %02x %02x %02x %02x",
                                        l, line[0], line[1], line[2], line[3], line[4], line[5], line[6], line[7]));
                     }
                  }
               }

               aqUsed += n;
               // while buffer contains a full AppLink message (>=7B and >=7+{length in B0..1}):
               while (aqUsed >= 7 && aqUsed >= 7 + AQREAD2(aqTail))
               {
                  const int dataLen = AQREAD2(aqTail);
                  const int command = AQREAD1(aqTail + 2) & 0x1f;
                  const int target = AQREAD1(aqTail + 2) >> 5;
                  const int connId = AQREAD4(aqTail + 3);
                  ETG_TRACE_COMP(("applinkproxy::Received message from SOCKS service: dataLen=%d command=%d target=%d connId=%d",
                                  dataLen, command, target, connId));
                  int clientSocket = -1;
                  // if message.command == 0 (keep-alive, 4.2.1):
                  if (command == 0)
                  {
                     ETG_TRACE_COMP(("applinkproxy::Received keep-alive "));
                  }
                  // elif message.command == 6 (information, 4.2.7):
                  else if (command == 6)
                  {
                     // process/store received information; compare protocol versions
                     if (dataLen < 1)
                     {
                        ETG_TRACE_SYS_MIN(("applinkproxy::Received too short 'information' command"));
                     }
                     else
                     {
                        int infoType = AQREAD1(aqTail + 7);
                        switch (infoType)
                        {
                           case 0: // protocol version
                              if (dataLen < 3)
                              {
                                 ETG_TRACE_SYS_MIN(("applinkproxy::Received too short 'information' command #%d", infoType));
                              }
                              else
                              {
                                 int major = AQREAD1(aqTail + 7 + 1);
                                 int minor = AQREAD1(aqTail + 7 + 2);
                                 ETG_TRACE_SYS(("applinkproxy::Remote protocol version: %d.%d", major, minor));
                              }
                              break;
                           case 1: // device ID
                              if (dataLen < 2)
                              {
                                 ETG_TRACE_SYS_MIN(("applinkproxy::Received too short 'information' command #%d", infoType));
                              }
                              else
                              {
                                 int length = AQREAD1(aqTail + 7 + 1);
                                 if (length > dataLen - 2)
                                 {
                                    length = dataLen - 2;
                                 }
                                 if (length > MAX_DEVICE_ID_LEN)
                                 {
                                    length = MAX_DEVICE_ID_LEN;
                                 }
                                 unsigned char remoteDeviceId[MAX_DEVICE_ID_LEN + 1]; // Zero-terminated
                                 for (int i = 0; i < length; i++)
                                 {
                                    remoteDeviceId[i] = AQREAD1(aqTail + 7 + 2 + i);
                                 }
                                 remoteDeviceId[length] = 0;
                                 onRemoteDeviceIdReceived(remoteDeviceId);
                                 serverDeviceId.assign((char*)remoteDeviceId);
                                 ETG_TRACE_SYS(("applinkproxy::Remote device ID: %s", serverDeviceId.c_str()));
                              }
                              break;
                           default:
                              ETG_TRACE_SYS_MIN(("applinkproxy::Received unknown InfoType: %d", infoType));
                              break;
                        }
                     }
                  }
                  else
                  {
                     // lookup associated client handle/socket, based on message.ConnectionID
                     for (int i = 0; i < socket_id_used; i++)
                     {
                        if (socket_id[i].id == connId)
                        {
                           clientSocket = socket_id[i].socket;
                           ETG_TRACE_COMP(("applinkproxy::Received message related to connId: %d "
                                           "clientSocket: %d", connId, clientSocket));
                           break;
                        }
                     }
                     if (clientSocket == -1)
                     {
                        // don't complain about missed 'close' messages
                        if (command != 5)
                        {
                           ETG_TRACE_SYS_MIN(("applinkproxy::Received message for unknown connId %d!", connId));
                        }
                     }
                  }
                  if (clientSocket != -1)
                  {
                     // if message.command == 2 (connect reply, 4.2.3):
                     if (command == 2)
                     {
                        ETG_TRACE_COMP(("applinkproxy::cmd=2 (connect reply, 4.2.3)"));

                        // log whether message.data indicates success (second byte is 0):
                        if (AQREAD1(aqTail + 7 + 1) == 0)
                        {
                           ETG_TRACE_SYS(("applinkproxy::SOCKS connection OK for connId %d", connId));
                        }
                        else
                        {
                           ETG_TRACE_SYS_MIN(("applinkproxy::SOCKS connect failed for connId %d "
                                              "with reply %d!", connId, (int) AQREAD1(aqTail + 7 + 1)));
                           // SOCKS service will follow up with a "close" command, which will clean up the socket and associations
                        }

                        // send message.data to associated SOCKS client
                        socketSend(clientSocket, aq, sizeof(aq), aqTail + 7, dataLen);
                     }
                     // if message.command == 3 (data, 4.2.4):
                     else if (command == 3)
                     {
                        ETG_TRACE_COMP(("applinkproxy::cmd=3 (data, 4.2.4)"));

                        // send message.data to associated SOCKS client
                        socketSend(clientSocket, aq, sizeof(aq), aqTail + 7, dataLen);
                        // send message to SOCKS service (data written, 4.2.5):
                        // {command=4, ConnectionId=message.ConnectionId, data=size}
                        char data[] = {(uint8_t)(dataLen >> 8), (uint8_t)(dataLen & 255)};
                        applinkSend(aApplinkHandle, 4, TARGET_PHONE, connId, data, sizeof(data));
                        ETG_TRACE_COMP(("applinkproxy::Forwarded data to SOCKS client for connId %d", connId));
                     }
                     // if message.command == 4 (data written, 4.2.5):
                     else if (command == 4)
                     {
                        ETG_TRACE_COMP(("applinkproxy::cmd=4 (data written, 4.2.5)"));

                        // no action required, but can be used for flow control
                     }
                     // if message.command == 5 (close, 4.2.6):
                     else if (command == 5)
                     {
                        ETG_TRACE_COMP(("applinkproxy::cmd=5 (close, 4.2.6)"));

                        ETG_TRACE_SYS(("applinkproxy::Client remotely disconnected for connId %d", connId));
                        // close client handle/socket
                        close(clientSocket);
                        // forget association between connection ID and client handle/socket
                        for (int i = 0; i < socket_id_used; i++)
                        {
                           if (socket_id[i].id == connId)
                           {
                              socket_id[i] = socket_id[socket_id_used - 1];
                              socket_id_used--;
                              break;
                           }
                        }
                     }
                     else
                     {
                        ETG_TRACE_SYS_MIN(("applinkproxy::Received unknown applink command %d "
                                           "for connId %d!", command, connId));
                     }
                  }
                  // remove message from read buffer
                  aqUsed -= 7 + dataLen;
                  aqTail += 7 + dataLen;
                  aqTail %= sizeof(aq);
               } // while
            }
            else
            {
               int clientSocket = fd;
               // lookup associated Connection ID, based on client handle/socket
               int connId = -1;
               for (int i = 0; i < socket_id_used; i++)
               {
                  if (socket_id[i].socket == clientSocket)
                  {
                     connId = socket_id[i].id;
                     ETG_TRACE_COMP(("applinkproxy::Received data from clientSocket: %d "
                                     "connId: %d", clientSocket, connId));
                     break;
                  }
               }
               if (connId == -1)
               {
                  ETG_TRACE_SYS_MIN(("applinkproxy::Received data from unknown SOCKS client!"));
                  continue;
               }
               // read data from socket; max data payload size for Applink is 65535 B
               char buffer[65535];
               int n = read(clientSocket, buffer, sizeof(buffer));
               // if data is available from a client:
               if (n > 0)
               {
                  // send message to SOCKS service (data, 4.2.4):
                  // {command=3, ConnectionId=<connection ID>, data=<data from client>}
                  applinkSend(aApplinkHandle, 3, TARGET_PHONE, connId, buffer, n);
                  ETG_TRACE_COMP(("applinkproxy::Forwarded data from SOCKS client for connId %d", connId));
               }
               // if the connection with a client is lost (n == 0) or an error occurred (n < 0):
               else
               {
                  if (n < 0)
                  {
                     ETG_TRACE_ERR(("applinkproxy::Error reading from SOCKS client: %s", strerror(errno)));
                  }

                  ETG_TRACE_SYS(("applinkproxy::Client locally disconnected for connId %d ", connId));
                  close(clientSocket);
                  clientSocket = -1;
                  // send message to SOCKS service (close, 4.2.6):
                  // {command=5, ConnectionId=<connection ID>, data=<empty>}
                  applinkSend(aApplinkHandle, 5, TARGET_PHONE, connId, NULL, 0);
                  // forget association between connection ID and client handle/socket
                  for (int i = 0; i < socket_id_used; i++)
                  {
                     if (socket_id[i].id == connId)
                     {
                        socket_id[i] = socket_id[socket_id_used - 1];
                        socket_id_used--;
                        break;
                     }
                  }
               }
            }
         }
      }
      if (isStopCalled)
      {
         ETG_TRACE_COMP(("applinkproxy::Stop applink proxy received!!!!"));

         char buf[5] = {0x00, 0x00};
         int n = read(pfd[0], buf, sizeof(buf));
         if ((n > 0) && !strcmp(buf, "stop"))
         {
            ETG_TRACE_COMP(("applinkproxy::Stop select message: %s", buf));

            //close all connection id
            for (int i = 0; i < socket_id_used; i++)
            {
               int connId = socket_id[i].id;
               ETG_TRACE_COMP(("applinkproxy::Sending close for connection id: %d", connId));
               applinkSend(aApplinkHandle, 5, TARGET_PHONE, connId, NULL, 0);
            }
         }
      }
   }

   //set gInterrupted to false, only after exiting the while to avoid race condition
   //when stop_applinkproxy() is called, gInterrupted is set to true.
   gInterrupted = false;
   signal(SIGINT, SIG_DFL); // Restore SIGINT and SIGPIPE handlers
   signal(SIGPIPE, SIG_DFL);

   ETG_TRACE_SYS(("applinkproxy::Quitting..."));

   for (int i = 0; i < socket_id_used; i++)
   {
      close(socket_id[i].socket);
   }
   close(socksServerSocket);
   close(pfd[0]);
   close(pfd[1]);
   return 0;
}


void stop_applinkproxy(void)
{
   ETG_TRACE_COMP(("applinkproxy::stop_applinkproxy()"));

   gInterrupted = true;
   char stopMsg[] = "stop";
   int bytesWritten = write(pfd[1], stopMsg, strlen(stopMsg));
   if (bytesWritten == -1)
   {
      ETG_TRACE_FATAL(("applinkproxy::stop_applinkproxy() write failed"));
   }
   else
   {
      ETG_TRACE_COMP(("applinkproxy::stop_applinkproxy() wrote: %d", bytesWritten));
   }
}


int setupStopSelect(void)
{
   ETG_TRACE_COMP(("applinkproxy::setupStopSelect()"));

   int ret = 0;

   if (pipe(pfd) == 0)
   {
      if (setBlockingModeEnabled(pfd[0], false) && setBlockingModeEnabled(pfd[1], false))
      {
         ETG_TRACE_COMP(("applinkproxy::setupStopSelect() SUCCESSFUL"));
         ret = 1;
      }
      else
      {
         ETG_TRACE_FATAL(("applinkproxy::setupStopSelect() FAILED"));
      }
   }
   else
   {
      ETG_TRACE_FATAL(("applinkproxy::setupStopSelect() Error in pipe creation: %d - %s", errno, strerror(errno)));
   }
   return ret;
}


// EOF
